<?php

namespace Studio\Totem\Http\Controllers;

use App\Models\App;
use App\Models\Project;
use App\Models\Website;
use Illuminate\Contracts\View\Factory;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;
use Studio\Totem\Contracts\TaskInterface;
use Studio\Totem\Events\TaskEvent;
use Studio\Totem\Http\Requests\TaskRequest;
use Studio\Totem\Http\Requests\TaskUpdateRequest;
use Studio\Totem\Task;
use Studio\Totem\Totem;

class TasksController extends Controller
{
    /**
     * @var TaskInterface
     */
    private TaskInterface $tasks;

    /**
     * TasksController constructor.
     *
     * @param  TaskInterface  $tasks
     */
    public function __construct(TaskInterface $tasks)
    {
        parent::__construct();

        $this->tasks = $tasks;
    }

    /**
     * Display a listing of the tasks.
     *
     * @return View
     */
    public function index(): View
    {
        return view('totem::tasks.index', [
            'tasks' => $this->tasks
                ->builder()
                ->sortableBy([
                    'description',
                    'last_ran_at',
                    'average_runtime',
                ], ['description'=>'asc'])
                ->when(request('q'), function (Builder $query) {
                    $query->where('description', 'LIKE', '%'.request('q').'%');
                })
                ->with('frequencies')
                ->paginate(20),
        ]);
    }

    /**
     * Show the form for creating a new task.
     *
     * @return View
     */
    public function create(): View
    {
        $website = Website::select(["name","id"])->get();

        $commands = Totem::getCommands()->map(function ($command) {
            return ['name' => $command->getName(), 'description' => $command->getDescription()];
        });

        return view('totem::tasks.form', [
            'task'          => new Task,
            'commands'      => $commands,
            'timezones'     => timezone_identifiers_list(),
            'frequencies'   => Totem::frequencies(),
            'websites' => $website
        ]);
    }

    /**
     * Store a newly created task in storage.
     *
     * @param  TaskRequest  $request
     * @return RedirectResponse
     */
    public function store(TaskRequest $request): RedirectResponse
    {
        $request["created_by"] = Auth::id();
        $store_task = $this->tasks->store($request->all());

        $store_task->update([
            "parameters" => $store_task->id,
            "created_by" => Auth::id()
        ]);
        return redirect()
            ->route('totem.tasks.all')
            ->with('success', trans('totem::messages.success.create'));
    }

    /**
     * Display the specified task.
     *
     * @param  Task  $task
     * @return Factory|View
     */
    public function view(Task $task)
    {
        return view('totem::tasks.view', [
            'task'  => $task,
        ]);
    }

    /**
     * Show the form for editing the specified task.
     *
     * @param  Task  $task
     * @return View
     */
    public function edit(Task $task): View
    {
        $website = Website::select(["name","id"])->get();

        $commands = Totem::getCommands()->map(function ($command) {
            return ['name' => $command->getName(), 'description' => $command->getDescription()];
        });

        return view('totem::tasks.form', [
            'task'          => $task,
            'commands'      => $commands,
            'timezones'     => timezone_identifiers_list(),
            'frequencies'   => Totem::frequencies(),
            'websites' => $website
        ]);
    }

    /**
     * Update the specified task in storage.
     *
     * @param  TaskRequest  $request
     * @param $task
     * @return RedirectResponse
     */
    public function update(TaskUpdateRequest $request, Task $task): RedirectResponse
    {
        $request["created_by"] = Auth::id();
        $task = $this->tasks->update($request->all(), $task);

        $task->update([
            "created_by" => Auth::id()
        ]);
        return redirect()->route('totem.task.view', $task)
            ->with('task', $task)
            ->with('success', trans('totem::messages.success.update'));
    }

    /**
     * Remove the specified task from storage.
     *
     * @param  Task  $task
     * @return RedirectResponse
     */
    public function destroy(Task $task)
    {
        $app = App::where("schedule_id",$task->id)->get();
        $task_string = (string)$task->id;
        $events = \App\Models\TaskEvent::where("task_id",$task->id)->get();
        $scLogs = \App\Models\AppsLog::where("task_id",$task->id)->get();
        $project = Project::whereJsonContains("task_id",$task_string)->get();


        if (count($app) > 0){
            return redirect()
                ->route('task.index')
                ->with('error', trans('This task related with apps you shold change the app before'));
        }else{
            if ($project) {
                $task_string = (string)$task->id;
                $dd = Project::whereJsonContains('task_id',$task_string)
                    ->update([
                        'task_id' => DB::raw("JSON_REMOVE(task_id, JSON_UNQUOTE(JSON_SEARCH(task_id, 'one', '$task_string')))")
                    ]);


            }
//            toastr()->error('This task related with apps you shold change the app before');
            foreach ($scLogs as $scLog){
                $scLog->delete();
            }
            foreach ($events as $event){
                $event->delete();
            }

            $this->tasks->destroy($task);

            return redirect()
                ->route('task.index')
                ->with('success', trans('totem::messages.success.delete'));
        }

    }
}
